import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.ServerSocket;
import java.net.Socket;

/**
 * Our Server class.  This class is responsible for establishing a connection to the Client, waiting for a simple message, turning the message
 * into upper case, and then replying to the Client with the new upper case message. 
 * 
 * @author Marc J. McDougall
 */
public class Server {
	
	//Public members that hold the Server port and ip address.  Note that we will simply use localhost for now.
	public static final int SERVER_PORT = 2000;
	public static final String IP_ADDR = "localhost";
	
	//Server socket that will simply accept any connection requests as they come in.
	private ServerSocket ss;
	
	//The socket that will provide the input/output streams for communication
	private Socket my_socket;
	
	//Object input and output streams that will be responsible for sending data between the Client and Server.
	private ObjectInputStream in;
	private ObjectOutputStream out;
	
	//Class member used to hold a reference to the most recent message.
	private String input;
	
	/**
	 * Default constructor.  Used to initialize the Server, wait for a connection from a Client, and then set up the data streams.
	 */
	public Server(){
		
		//Initialized the input message to the null String
		input = "";
		
		try{
			
			//Initializes the ServerSocket and tells it to listen on the port described above.
			ss = new ServerSocket(SERVER_PORT);
			
			try{
				
				//Inform the operator that the Server is waiting for a connection
				System.out.println("Waiting for connection...");
				
				//Immediately accept a connection when one is found.  Note, there is a very bad practice in terms of security.  You never want to just accept() connections
				// and assume they are not malicious!
				my_socket = ss.accept();
				
				//Inform the operator that a connection has been detected
				System.out.println("**Connection Detected**");
				
				//Initialize the streams
				out = new ObjectOutputStream(my_socket.getOutputStream());
				in = new ObjectInputStream(my_socket.getInputStream());
				
				//Inform the operator that the streams have been established
				System.out.println("**Streams Established**");
			}
			catch(IOException e){
				
				//Do nothing here, just print the stack trace.
				e.printStackTrace();
			}
		}
		catch(IOException e){
			
			//Do nothing here, just print the stack trace.
			e.printStackTrace();
		}
	}
	
	/**
	 * Method that will contain everything that must be done with the Server.
	 **/
	public void run(){
		
		try{
			
			//Wait for input from the stream (a message sent from the Client)
			input = (String) in.readObject();
			
			//Inform the operator that some input has been received.
			System.out.println("**Input Received**");
			
			//Write the object in upper case
			out.writeObject(input.toUpperCase());
			
			//Flush the stream to ensure that the data is sent immediately
			out.flush();
		} 
		catch (ClassNotFoundException e){
			
			//Do nothing here, just print the stack trace.
			e.printStackTrace();
		}
		catch (IOException e) {
			
			//Do nothing here, just print the stack trace.
			e.printStackTrace();
		}
	}
	
	/**
	 * Main method that will be called at the start of the execution.  Simply creates a new Server and calls run() repeatedly.
	 * 
	 * @param args Default arguments for the Main method - unused in our example.
	 */
	public static void main(String[] args){
		
		Server s = new Server();
		
		while(true){
			
			s.run();
		}
	}
}
